from enum import Enum

class TokenType(Enum):
    terminator  = 0
    int_token   = 1
    str_token   = 2
    EndOfStream = 3

class Token:
    def __init__(self, token_type, token_value = ';'):
        self.token_type  = token_type
        self.token_value = token_value

    def isTerminator(self):
        return self.token_type == TokenType.terminator

    def isInteger(self):
        return self.token_type == TokenType.int_token

    def isString(self):
        return self.token_type == TokenType.str_token

    def isEndOfStream(self):
        return self.token_type == TokenType.EndOfStream

    def value(self):
        return self.token_value
    
    def __repr__(self):
        if self.isTerminator():
            return 'terminator'
        elif self.isInteger():
            return 'integer = ' + str(self.token_value)
        elif self.isString():
            return 'string  = ' + self.token_value
        else:   # isEndOfStream()
            return 'end_of_stream'

class State(Enum):
    looking = 1
    zero    = 2
    integer = 3
    string  = 4
    
def scan(program):
    token = []
    accum = ''

    # the state machine
    state = State.looking
    pos   = 0
    while pos < len(program):
        # print(pos, ' ', state)           # trace flow
        if state == State.looking:
            if program[pos].isspace():
                pass
            elif program[pos] == ';':
                token.append( Token(TokenType.terminator) )
            elif program[pos] in '0':
                accum = program[pos]
                state = State.zero
            elif program[pos] in '123456789':
                accum = program[pos]
                state = State.integer
            else:
                accum = program[pos]
                state = State.string
            pos += 1
        elif state == State.zero:
            if program[pos].isspace():
                token.append( Token(TokenType.integer, 0) )
            elif program[pos] == ';':
                token.append( Token(TokenType.integer, 0) )
                token.append( Token(TokenType.terminator) )
            elif program[pos].isdigit():
                error_msg ='Integers do not have leading zeros: 0{}'
                raise ValueError(error_msg.format(program[pos]))
            else:
                error_msg ='Invalid character after a 0: 0{}'
                raise ValueError(error_msg.format(program[pos]))
            accum = ''
            state = State.looking
            pos  += 1
        elif state == State.integer:
            if program[pos].isdigit():
                accum += program[pos]
            elif program[pos].isspace():
                token.append( Token(TokenType.int_token, int(accum)) )
                accum = ''
                state = State.looking
            elif program[pos] == ';':
                token.append( Token(TokenType.int_token, int(accum)) )
                token.append( Token(TokenType.terminator) )
                accum = ''
                state = State.looking
            else:
                error_msg ='Invalid character in integer {}*{}*'
                raise ValueError(error_msg.format(accum, program[pos]))
            pos += 1
        elif state == State.string:
            if program[pos].isspace():
                token.append( Token(TokenType.str_token, accum) )
            elif program[pos] == ';':
                token.append( Token(TokenType.str_token, accum) )
                token.append( Token(TokenType.terminator) )
            else:
                error_msg ='All strings are single character {}*{}*'
                raise ValueError(error_msg.format(accum, program[pos]))
            accum = ''
            state = State.looking
            pos  += 1
        else:
            error_msg ='Invalid state {}.  How did that happen?'
            raise TypeError(error_msg.format(state))

    # handle accumulator at end of file
    if accum:
        if state == State.zero:
            token.append( Token(TokenType.integer, 0) )
        elif state == State.integer:
            token.append( Token(TokenType.int_token, int(accum)) )
        elif state == State.string:
            token.append( Token(TokenType.str_token, accum) )
        else:
            error_msg ='Invalid state {} with this accum {}'
            raise TypeError(error_msg.format(state, accum))

    # return list of tokens
    token.append( Token(TokenType.EndOfStream) )
    return token

if __name__ == "__main__":
    programs = ['', ' ', ';', '6 x;', '3 9 x;\n6 3 b 3 X 3 b;\n3 9 x;']
    for p in programs:
      print( '*{}* --- {}'.format(p, scan(p)) )
